<?php
/**
 * 2018 Cobiro.
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    RST-IT
 * @copyright Cobiro 2018
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

require_once _PS_MODULE_DIR_ . 'cobiro/cobiro.php';

/**
 * Class WebserviceSpecificManagementCobiro.
 */
class WebserviceSpecificManagementCobiro implements WebserviceSpecificManagementInterface
{
    const RESOURCE_GMC = 'gmc';
    const RESOURCE_GCV = 'gcv';

    private $objOutput;
    private $wsObject;

    /**
     * Returns Content.
     *
     * @return string
     */
    public function getContent()
    {
        // The API (which calls this endpoint) does not need any response, so we simply return an empty string.
        return '';
    }

    /**
     * Sets Object output.
     *
     * @param WebserviceOutputBuilderCore $obj
     *
     * @return $this
     */
    public function setObjectOutput(WebserviceOutputBuilderCore $obj)
    {
        $this->objOutput = $obj;

        return $this;
    }

    /**
     * Returns WS object.
     *
     * @return WebserviceRequest
     */
    public function getWsObject()
    {
        return $this->wsObject;
    }

    /**
     * Sets Webservice object.
     *
     * @param WebserviceRequestCore $obj
     *
     * @return $this
     */
    public function setWsObject(WebserviceRequestCore $obj)
    {
        $this->wsObject = $obj;

        return $this;
    }

    /**
     * Returns Object output.
     *
     * @return WebserviceOutputBuilder
     */
    public function getObjectOutput()
    {
        return $this->objOutput;
    }

    /**
     * Main method for Webservice class.
     *
     * Fixed json format
     */
    public function manage()
    {
        $resource = Tools::getValue('resource');

        switch ($resource) {
            case static::RESOURCE_GMC:
                $this->handleGMC();
                break;
            case static::RESOURCE_GCV:
                $this->handleGCV();
                break;
            default:
                $this->handleModuleInfo();
                break;
        }
    }

    protected function handleGMC()
    {
        $requestJson = $this->requestJson();
        $this->validate('key', $requestJson);
        Configuration::updateValue(Cobiro::OPT_GMC, $requestJson['key']);

        http_response_code(204);
    }

    protected function requestJson()
    {
        if ($_SERVER['REQUEST_METHOD'] === 'GET') {
            $this->exception('Method Not Allowed', 405);
        }

        return Tools::jsonDecode(Tools::file_get_contents('php://input'), true);
    }

    /**
     * Returns Exception if form of json response.
     *
     * @param null $msg
     * @param int  $code
     */
    public function exception($msg = null, $code = 400)
    {
        http_response_code($code);
        $this->printJsonAndExit(array('code' => $code, 'msg' => $msg ?: 'Bad Request'));
    }

    protected function printJsonAndExit($array)
    {
        header('Content-Type: application/json');
        print Tools::jsonEncode($array);
        // Exit to avoid PrestaShop webservice output.
        exit;
    }

    protected function validate($paramName, $data = null)
    {
        if ($data) {
            if (!array_key_exists($paramName, $data)) {
                $this->exception('Param is required! :' . $paramName, 400);
            }

            return $data[$paramName];
        }

        $this->exception('Unprocessable Entity', 422);
    }

    protected function handleGCV()
    {
        $requestJson = $this->requestJson();

        $id = $this->validate('id', $requestJson);
        $label = $this->validate('label', $requestJson);

        Configuration::updateValue(Cobiro::OPT_GCV_ID, $id);
        Configuration::updateValue(Cobiro::OPT_GCV_LABEL, $label);

        http_response_code(204);
    }

    protected function handleModuleInfo()
    {
        $this->printJsonAndExit(array(
            'shop_version'   => _PS_VERSION_,
            'module_version' => Module::getInstanceByName(Cobiro::NAME)->version,
        ));
    }
}
