<?php
/**
 * 2018 Cobiro.
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    RST-IT
 * @copyright Cobiro 2018
 * @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

/**
 * Class InitService The main install and uninstall procedures for Cobiro Module.
 */
class InitService
{
    /**
     * Cobiro Module.
     *
     * @var Cobiro
     */
    private $module;

    /**
     * InitService constructor.
     *
     * @param Cobiro $module
     */
    public function __construct(Cobiro $module)
    {
        $this->module = $module;
    }

    /**
     * Installs module.
     */
    public function install()
    {
        require_once _PS_MODULE_DIR_ . 'cobiro/cobiro/helpers/CobiroTools.php';

        if (Shop::isFeatureActive()) {
            Shop::setContext(Shop::CONTEXT_ALL);
        }

        if (!$this->module->registerHook('displayHeader') ||
            !$this->module->registerHook('actionAdminControllerSetMedia') ||
            !$this->module->registerHook('displayOrderConfirmation') ||
            !$this->installTab() ||
            !$this->installModuleAPI()
        ) {
            return false;
        }
        $this->clearCache();

        return true;
    }

    /**
     * Installs tab for Cobiro module.
     *
     * @return int
     */
    private function installTab()
    {
        require_once _PS_MODULE_DIR_ . 'cobiro/cobiro/helpers/CobiroTools.php';
        try {
            $root_tab = new Tab();
            $root_tab->active = 1;
            $root_tab->class_name = 'Admin';
            $root_tab->name = array();
            foreach (Language::getLanguages(true) as $lang) {
                $root_tab->name[$lang['id_lang']] = 'Cobiro';
            }
            $root_tab->module = Cobiro::NAME;
            $root_tab->id_parent = 0;
            $root_tab->save();

            $tab = new Tab();
            $tab->active = 1;
            $tab->class_name = 'AdminCobiro';
            $tab->name = array();
            foreach (Language::getLanguages(true) as $lang) {
                $tab->name[$lang['id_lang']] = 'Configure';
            }
            $tab->module = Cobiro::NAME;
            $tab->id_parent = $root_tab->id;

            $tab->save();

            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Installs module API layer in prestashop.
     */
    private function installModuleAPI()
    {
        Configuration::updateGlobalValue('PS_WEBSERVICE', true);

        $token = mb_strtoupper(Tools::passwdGen(32));
        try {
            Db::getInstance()->insert('webservice_account', array(
                'key'         => pSQL($token),
                'description' => pSQL('Cobiro Access'),
                'class_name'  => pSQL('WebserviceRequest'),
                'is_module'   => (int) 0,
                'module_name' => pSQL(Cobiro::NAME),
                'active'      => (int) 1,
            ));

            $wsAccountId = Db::getInstance()->Insert_ID();

            // Install for all shops.
            $allShops = Shop::getCompleteListOfShopsID();
            foreach ($allShops as $shopId) {
                Db::getInstance()->insert('webservice_account_shop', array(
                    'id_webservice_account' => $wsAccountId,
                    'id_shop' => $shopId,
                ));
            }

            $this->installPermissions($wsAccountId);
            $this->installPermissions140($wsAccountId);

            return true;
        } catch (PrestaShopDatabaseException $e) {
            return false;
        }
    }

    public function clearCache()
    {
        $this->clearCacheObjects();

        $class_index_path = $this->getClassIndexFile();
        unlink($class_index_path);
    }

    private function getClassIndexFile()
    {
        if (version_compare(_PS_VERSION_, '1.7', '<')) {
            return _PS_ROOT_DIR_ . '/cache/class_index.php';
        }

        if (defined('_PS_MODE_DEV_') && _PS_MODE_DEV_) {
            return _PS_ROOT_DIR_ . '/app/cache/dev/class_index.php';
        }

        return PS_ROOT_DIR_ . '/app/cache/prod/class_index.php';
    }

    /**
     * Uninstalls module.
     */
    public function uninstall()
    {
        require_once _PS_MODULE_DIR_ . 'cobiro/cobiro/helpers/CobiroTools.php';

        if (!$this->uninstallOptions() ||
            !$this->uninstallTab() ||
            !$this->uninstallModuleAPI()
        ) {
            return false;
        }
        $this->clearCache();

        return true;
    }

    /**
     * Uninstall CobiroModule options.
     */
    private function uninstallOptions()
    {
        $status = true;

        $status = $status && $this->uninstallOption(Cobiro::OPT_GCV_LABEL);
        $status = $status && $this->uninstallOption(Cobiro::OPT_GCV_ID);
        $status = $status && $this->uninstallOption(Cobiro::OPT_GMC);

        return $status;
    }

    public function uninstallOption($name)
    {
        $value = Configuration::get($name);

        if (!$value) {
            return true;
        }

        try {
            Configuration::deleteByName($name);

            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Uninstalls Cobiro tab.
     */
    private function uninstallTab()
    {
        $tabId = Tab::getIdFromClassName(Cobiro::ADMIN_CONTROLLER);
        if ($tabId) {
            $tab = new Tab($tabId);
            $parentId = $tab->id_parent;

            if ($parentId) {
                $tabParent = new Tab($parentId);
            }

            $tab->delete();

            if ($parentId) {
                $tabParent->delete();
            }
        }

        return true;
    }

    /**
     * Uninstalls module API layer in prestashop.
     */
    private function uninstallModuleAPI()
    {
        require_once _PS_MODULE_DIR_ . 'cobiro/cobiro/services/RepositoryService.php';

        try {
            Db::getInstance()->delete('webservice_permission', "resource = '". pSQL(Cobiro::NAME) . "'");
            $webserviceAccountId = RepositoryService::instance()->getPrestaWebserviceAccountId();

            if ($webserviceAccountId) {
                Db::getInstance()->delete('webservice_account_shop', "id_webservice_account = {$webserviceAccountId}");
                Db::getInstance()->delete('webservice_account', "id_webservice_account = {$webserviceAccountId}");
            }

            return true;
        } catch (PrestaShopDatabaseException $e) {
            return false;
        }
    }

    private function clearCacheObjects()
    {
        if (version_compare(_PS_VERSION_, '1.5', '>=') &&
            version_compare(_PS_VERSION_, '1.6', '<')
        ) {
            Tools::clearSmartyCache();
            Media::clearCache();
            Autoload::getInstance()->generateIndex();

            return;
        }

        Tools::clearSmartyCache();
        Tools::clearXMLCache();
        Media::clearCache();
        Tools::generateIndex();
    }

    /**
     * Install permissions
     * @param int $wsAccountID
     */
    private function installPermissions($wsAccountID)
    {
        $this->installPermission(Cobiro::NAME, 'GET', $wsAccountID);
        $this->installPermission(Cobiro::NAME, 'POST', $wsAccountID);
        $this->installPermission('products', 'GET', $wsAccountID);
        $this->installPermission('categories', 'GET', $wsAccountID);
        $this->installPermission('orders', 'GET', $wsAccountID);
    }

    /**
     * Install permissions need for version 1.4.0
     * @param int $wsAccountID
     */
    public function installPermissions140($wsAccountID)
    {
        $this->installPermission('product_options', 'GET', $wsAccountID);
        $this->installPermission('product_option_values', 'GET', $wsAccountID);
        $this->installPermission('combinations', 'GET', $wsAccountID);
        $this->installPermission('languages', 'GET', $wsAccountID);
        $this->installPermission('order_states', 'GET', $wsAccountID);
        $this->installPermission('tax_rule_groups', 'GET', $wsAccountID);
        $this->installPermission('tax_rules', 'GET', $wsAccountID);
        $this->installPermission('taxes', 'GET', $wsAccountID);
        $this->installPermission('stock_availables', 'GET', $wsAccountID);
        $this->installPermission('manufacturers', 'GET', $wsAccountID);
    }

    /**
     * Install specific permission
     * @param string $resource
     * @param string $method
     * @param int    $wsAccountID
     */
    private function installPermission($resource, $method, $wsAccountID)
    {
        Db::getInstance()->insert('webservice_permission', array(
            'resource'              => pSQL($resource),
            'method'                => pSQL($method),
            'id_webservice_account' => $wsAccountID,
        ));
    }
}
